<?php

namespace App\Console\Commands;

use App\Imports\TiersInfosSaliariesImport;
use App\Models\ST\auto\action_interne;
use App\Models\ST\auto\aide_action_sts;
use App\Models\ST\auto\tiers_infos_salaries;
use App\Models\TiersInfosSaliariesTemp;
use App\Services\ApiLogManager;
use App\Services\RhImportService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Ramsey\Uuid\Uuid;
use Spatie\Emoji\Emoji;

class ImportInfoSalarieFromCsvCommand extends Command
{
    /**
     * TContexte
     * Les données sociales sont gérées sur le logiciel STS.
     *
     * Nous exportons dans STS via API les données suivantes :
     * - Les nouveaux salariés
     * - Les visites médicales et obligatoires.
     * Nous souhaitons importer depuis STS via fichier pla tous les jours à 12h30 et 23h30
     * - Les autres évènements influant sur la paie.
     *
     * L’idée est de renvoyer vers le TMS pour les exploitants l’ensemble des données sociales :
     * - Les nouveaux salariés
     * - Leurs indisponibilités (visites et événements).
     *
     * Besoin
     * • En tant que : exploitant et RH
     * • Je veux : importer les évènements influant sur paie depuis STS vers E Global Pro deux fois par jours
     * • Afin de : ensuite pouvoir les redistribuer vers XYRICS.
     *
     * Précisions
     * • BDD cible : bdd_trs_pdl
     * • Les tables cibles sont :
     * ▪ Tiers
     * ▪ Tiers_infos_salaries
     *
     * • Lors de l’import des salariés dans STS depuis notre API, ils importent également le PID_TIERS du salarié
     * depuis la table tiers_infos_salaries
     *
     * • Ils déposent par la suite un fichier type qui :
     * o Exporte tous les évènements des mois “non clôturés” dans STS (3 mois pour PDL. Il faut donc faire
     * un dédoublonnage automatique sur : le PID_TIERS, la date de début et de fin avant d’importer) via
     * un champs concat par exemple.
     * o Les colonnes et règle d’imports sont les suivantes (Attention les colonnes n’ont pas d’entête, il faut
     * donc se baser sur le numéro).
     *
     * @var string
     */
    protected $signature = 'salaries:import-info-from_csv {--clear-temps} {--no-archive}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Importe les données des salariés depuis csv';

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $this->output->title('Importation des évènements RH depuis le fichier csv');

        // option pour la dev qui permet de refaire passer un csv
        if ($this->option('clear-temps')) {
            TiersInfosSaliariesTemp::query()->truncate();
        }


        $files = Storage::disk('sts_rh_import')->files("/");
        $fulltable = [];


        foreach ($files as $file) {

            $started_at = now()->format('Y-m-d H:i:s');
            ob_start();
            $start = microtime(true);

            $total_imported_for_db = 0;

            $total_previously_imported_for_db = 0;
            $total_suppressed_for_db = 0;


            $fullpath = Storage::disk('sts_rh_import')->path($file);
            $this->info(Emoji::backhandIndexPointingRight() . " Traitement de {$file}");

            Excel::import(new TiersInfosSaliariesImport(), $fullpath);

            //Tous ceux dont la date est supérieure au début du dernier import.

            //dédoublonnage automatique sur : le PID_TIERS, la date de début et de fin avant d’importer) via
            //un champs concat par exemple.
            $imported_csv_lines = TiersInfosSaliariesTemp::query()->where("created_at", ">=", $started_at)->get();

            $progressBar = $this->output->createProgressBar($imported_csv_lines->count());
            $progressBar->setBarCharacter('=');
            $progressBar->setProgressCharacter(Emoji::hourglassNotDone());
            $progressBar->setEmptyBarCharacter(' ');
            $progressBar->start();

            // A ce stade il ne doit rester que des lignes a importer.
            foreach ($imported_csv_lines as $csv_line) {
                $progressBar->advance();

                if ($this->checkImportableLine($csv_line)) {


                    // Traitement post import sur les lignes crées dans tiers_infos_salaries


                    // SUPPRESION ACTION INTERNE
                    if ($csv_line->type == 'S') {
                        // mark event as suppressed dans action type.
                       // $this->addRhEvent($csv_line);
                        //$this->line((new RhImportService($csv_line))->deleteRhEvent());
                        (new RhImportService($csv_line))->deleteRhEvent();
                        $total_suppressed_for_db++;
                    }

                    // CREATION ACTION INTERNE
                    if ($csv_line->type == 'C') {
                       // $this->delRhEvent($csv_line);
                        $this->line((new RhImportService($csv_line))->createRhEvent());
                        (new RhImportService($csv_line))->createRhEvent();
                        $total_imported_for_db++;
                    }
                } else {
                    // On ne fait
                    $total_previously_imported_for_db++;
                }
            }

            // timer end + Log $total_importde_for_db
            $end = microtime(true);
            $phpinfo = ob_get_contents();
            ob_end_clean();
            $time = $end - $start;
            $finished_at = now()->format('Y-m-d H:i:s');


            $this->output->newLine();


            $progressBar->finish();
            $this->output->newLine(2);


            if ($this->option('no-archive')) {
                $archived = "no-archive option activatée";
            } else {
                // Archivage du fichier (besoin de savoir sur quel serveur => storage a crée )
                $archived = (new RhImportService(null))->archiveCsvRhInfosFile($file);
            }


            $fulltable[] = [$file, $total_previously_imported_for_db, $total_suppressed_for_db, $total_imported_for_db, $archived];

            (new ApiLogManager())
                ->log(
                    "import",
                    "RH",
                    $started_at,
                    $finished_at,
                    $total_imported_for_db,
                    $time,
                    [
                        "file" => $file,
                        "allready_done" => $total_previously_imported_for_db,
                        "deleted" => $total_suppressed_for_db,
                        "added" => $total_imported_for_db,
                        "started_at" => $started_at,
                        "finished_at" => $finished_at,
                        "archived" => $archived,
                    ],
                );



            // présence mod-php ou pas ?


        }


        /* Affiche un table résumé. */
        $headers = ["Fichier", "Déjà fait", "Supprimés", "Ajoutés", "Archivé"];
        $this->table($headers, $fulltable);


        // Traitement post import sur les lignes crées dans tiers_infos_salaries
        //

        return Command::SUCCESS;
    }

    public function checkImportableLine(TiersInfosSaliariesTemp $csv_line): bool
    {
       // $importable = true;

        /*
         *
         * [
            'matricule_de_paie',
            'pid_tiers',
            'code_evenement',
            'date_de_debut',
            'demi_journee_de_debut',
            'date_de_fin',
            'demi_journee_de_fin',
            'type',
            'controle'
        ]



            si col=4 (Demi journée de début) si M .... HERE_DEBUT=8 si A HEURE_DEBUT=13
            si col=5 (dateFin) si vide pas import
            si col=6 (Demi journée de fin) si M .... HERE_FIN=12 si A HEURE_FIN=18
            col=7 (type) si vide pas import C => créer la ligne
        */

//            si col=0 (Matricule de paie) est vide pas d'import
        if ($csv_line->matricule_de_paie == '' || $csv_line->matricule_de_paie == null) $importable = false;

//            si col=1 (PID tiers) n'a pas de correspondance: pas d'import
        $tiers = tiers_infos_salaries::find($csv_line->pid_tiers);
        if ($tiers == null) $importable = false;

//            si col=3 (dateDebut) vide ne pas importer
        if ($csv_line->date_de_debut == '' || $csv_line->date_de_debut == null) $importable = false;
        if ($csv_line->date_de_fin == '' || $csv_line->date_de_fin == null) $importable = false;

        //test du champ de controle, il peut peut être présent qu'une seule fois

        $nbr_elements = TiersInfosSaliariesTemp::query()
            ->where('controle', $csv_line->controle)
            ->count();

        if ($nbr_elements >= 1) $importable = false;


        //    if ($csv_line->type == 'S') $importable = true; // car une action à faire (dernier refait plus tard)


        return $importable;
    }


    public function addRhEvent(TiersInfosSaliariesTemp $csv_line){
        /*[
            'matricule_de_paie',
            'pid_tiers',
            'code_evenement',
            'date_de_debut',
            'demi_journee_de_debut',
            'date_de_fin',
            'demi_journee_de_fin',
            'type',
            'controle'
        ]*/

        $id_action_interne = Uuid::uuid4();


        $tier_infos_salarie = tiers_infos_salaries::query()
            ->where('PID_TIERS',$csv_line->pid_tiers)
            ->whereNull('DATE_SORTIE')
            ->orderBy('DATE_ENTREE')
        ->first();

        $nom_dossier = $tier_infos_salarie->NOM_DOSSIER;
        $n_dossier = $tier_infos_salarie->N_DOSSIER;
        $pid_tiers_infos_salaries = $tier_infos_salarie->ID_TIERS_INFOS_SALARIES;
        $raison_social = $tier_infos_salarie->NOM_SALRIE;
        $type_evenement = "Evènement";
        $prise_rdvs = "OK";
        $memo = "Importé depuis STS le ".now()->format('Y-m-d H:i:s');




// = concat des lignes de l’import sauf de la
//dernière colonne
        $concat = "";

        $type_action = aide_action_sts::query()
            ->where('CODE_ACTION', $csv_line->code_evenement)
            ->first();



        action_interne::create(
            [
                'A_FAIRE_PAR' => null,
                'DATE_DEBUT' => null,
                'DATE_FIN' => null,
                'FAIT' => null,
                'FAMILLE' => null,
                'FIMO' => null,
                'HEURE_DEBUT' => null,
                'HEURE_FIN' => null,
                'ID_ACTION_INTERNE' => $id_action_interne,
                'MEMO' => null,
                'NOMBRE' => null,
                'NOM_DOSSIER' => $nom_dossier,
                'N_DOSSIER' => $n_dossier,
                'PID_TIERS' => null,
                'PID_TIERS_INFOS_SALARIES' => $pid_tiers_infos_salaries,
                'RAISON_SOCIALE' => $raison_social,
                'SYS_DATE_CREATION' => null,
                'SYS_DATE_MODIFICATION' => null,
                'SYS_HEURE_CREATION' => null,
                'SYS_HEURE_MODIFICATION' => null,
                'SYS_USER_CREATION' => null,
                'SYS_USER_MODIFICATION' => null,
                'TYPE_ACTION' => null,
                'TYPE_EVENEMENT' => null,
                'VISITE_APTH' => null,
                'VM_DATE_VALIDITE' => null,
                'VM_ENTRETIEN_INFIRMIER' => null,
                'VM_ENTRETIEN_MEDECI' => null,
                'ABS_NR' => null,
                'ABS_REM' => null,
                'DATE_HEURE' => null,
                'HEURE_CONDUITE' => null,
                'HEURE_CONGES_PAYES' => null,
                'HEURE_DELEGATION' => null,
                'HEURE_DIPSO_DE' => null,
                'HEURE_DISPO_SOLO' => null,
                'HEURE_FORMATION' => null,
                'HEURE_JOURS_FERIES' => null,
                'HEURE_MALADIE' => null,
                'HEURE_NUIT' => null,
                'HEURE_REPOS_COMPENSATEUR' => null,
                'HEURE_SERVICE' => null,
                'HEURE_TRAVAIL' => null,
                'KM' => null,
                'LIEU_VISITE_MEDICALE' => null,
                'MEDECIN' => null,
                'N_DEJ' => null,
                'N_DEJ_INTER' => null,
                'N_DEPART_DIMANCHE' => null,
                'N_DEPART_FERIE' => null,
                'N_DIN' => null,
                'N_DIN_INTER' => null,
                'N_DIVERS' => null,
                'N_IMPORT' => null,
                'N_NUIT' => null,
                'N_NUIT_INTER' => null,
                'N_PDJ' => null,
                'N_PDJ_INTER' => null,
                'N_SAMEDI_APRES_18_LUNDI_MATIN' => null,
                'N_SAMEDI_MATIN_LUNDI_MATIN' => null,
                'N_TRAVAIL_JOUR_FERIE_COMPLET' => null,
                'N_UNI' => null,
                'N_UNI_INTER' => null,
                'N_VENDREDI_19_SAMEDI_APRES_19' => null,
                'N_VENDREDI_SOIR_LUNDI_MATIN' => null,
                'PID_TIERS_INFO_PAIE' => null,
                'SERVICE_ASSIMILE' => null,
                'T_DEJ' => null,
                'T_DEJ_INTER' => null,
                'T_DEPART_DIMANCHE' => null,
                'T_DEPART_FERIE' => null,
                'T_DIN' => null,
                'T_DIN_INTER' => null,
                'T_DIVERS' => null,
                'T_NUIT' => null,
                'T_NUIT_INTER' => null,
                'T_PDJ' => null,
                'T_PDJ_INTER' => null,
                'T_SAMEDI_APRES_18_LUNDI_MATIN' => null,
                'T_SAMEDI_MATIN_LUNDI_MATIN' => null,
                'T_TRAVAIL_JOUR_FERIE_COMPLET' => null,
                'T_UNI' => null,
                'T_UNI_INTER' => null,
                'T_VENDREDI_19_SAMEDI_APRES_19' => null,
                'T_VENDREDI_SOIR_LUNDI_MATIN' => null,
                'VITESSE_MOYENNE' => null,
                'HEURE_SERV' => null,
                'JOUR_FERIE' => null,
                'MATRICULE' => null,
                'HEURE_ACCIDENT' => null,
                'TEMPS_ALERTE' => null,
                'FREQUENCE' => null,
                'HEURE_JOUR' => null,
                'HEURE_AMPLITUDE' => null,
                'HEURE_INTEGRATION' => null,
                'N_DEPART_DIMANCHE_WE' => null,
                'N_DIVERS_INTER' => null,
                'N_TICKET_RESTAU' => null,
                'T_DEPART_DIMANCHE_WE' => null,
                'T_DIVERS_INTER' => null,
                'T_TICKET_RESTAU' => null,
                'RH_ATTRIBUE' => null,
                'DATE_ALERTE' => null,
                'CODE_JOUR' => null,
                'SUP' => null,

            ]
        );
    }

    public function delRhEvent(TiersInfosSaliariesTemp $csv_line){
        /*[
            'matricule_de_paie',
            'pid_tiers',
            'code_evenement',
            'date_de_debut',
            'demi_journee_de_debut',
            'date_de_fin',
            'demi_journee_de_fin',
            'type',
            'controle'
        ]*/



    }

}
